using System;
using System.IO;
using System.Collections.Generic;

namespace ASFBufferFix {
	class ASFParse {
		private string _path;
		private FileStream _fs;
		private BinaryReader _br;
		private Dictionary<Guid, string> _guidToName;
		private Dictionary<string, Guid> _guidByName;
		private List<long> _extendedStreamPropertiesDataOffsets;

		public ASFParse(string path) {
			InitGuids();

			_path = path;
			_fs = new FileStream(path, FileMode.Open, FileAccess.Read, FileShare.Read);
			_br = new BinaryReader(_fs);

			_extendedStreamPropertiesDataOffsets = new List<long>();

			ParseMain(_fs.Length, 0);

			_br.Close();
			_fs.Close();
		}

		public int FixBufferSizes() {
			FileStream fs = new FileStream(_path, FileMode.Open, FileAccess.ReadWrite, FileShare.Read);
			BinaryWriter bw = new BinaryWriter(fs);

			foreach (long pos in _extendedStreamPropertiesDataOffsets) {
				bw.BaseStream.Seek(pos, SeekOrigin.Begin);
				bw.Seek(20, SeekOrigin.Current);
				bw.Write((uint)4000);
				bw.Seek(8, SeekOrigin.Current);
				bw.Write((uint)4000);
			}

			bw.Close();
			fs.Close();

			return _extendedStreamPropertiesDataOffsets.Count;
		}

		private string GetGuidDescription(Guid guid) {
			string desc;
			return _guidToName.TryGetValue(guid, out desc) ? desc : guid.ToString();
		}

		private void ParseMain(long objEnd, int level) {
			Guid objGuid;
			UInt64 objSize;
			int objIndex;
			Int64 nextObjPosition;

			objIndex = 0;

			while (_br.BaseStream.Position < objEnd) {
				ReadObjectHeader(out objGuid, out objSize);

				if ((level == 0) && (objIndex == 0) && (objGuid != _guidByName["ASF_Header_Object"])) {
					throw new Exception("This is not an ASF file.");
				}

				Console.WriteLine(new string(' ', level * 2) + "GUID: " +
					GetGuidDescription(objGuid));

				nextObjPosition = _br.BaseStream.Position + (long)objSize;

				if (objGuid == _guidByName["ASF_Header_Object"]) {
					_br.ReadBytes(6);
					ParseMain(nextObjPosition, level + 1);
				}
				else if (objGuid == _guidByName["ASF_Header_Extension_Object"]) {
					_br.ReadBytes(22);
					ParseMain(nextObjPosition, level + 1);
				}
				else if (objGuid == _guidByName["ASF_Extended_Stream_Properties_Object"]) {
					_extendedStreamPropertiesDataOffsets.Add(_br.BaseStream.Position);
				}

				_br.BaseStream.Seek(nextObjPosition, SeekOrigin.Begin);
				objIndex++;
			}
		}

		private void ReadObjectHeader(out Guid objGuid, out UInt64 objSize) {
			objGuid = ReadGuid();
			objSize = ReadUInt64() - 24;
		}

		private Guid ReadGuid() {
			return new Guid(_br.ReadBytes(16));
		}

		private UInt64 ReadUInt64() {
			return _br.ReadUInt64();
		}

		private void AddGuid(string name, string guid) {
			Guid g = new Guid(guid);
			_guidToName.Add(g, name);
			_guidByName.Add(name, g);
		}

		private void InitGuids() {
			_guidToName = new Dictionary<Guid, string>();
			_guidByName = new Dictionary<string, Guid>();

			// Top-Level ASF Object
			AddGuid("ASF_Header_Object", "75B22630-668E-11CF-A6D9-00AA0062CE6C");
			AddGuid("ASF_Data_Object", "75B22636-668E-11CF-A6D9-00AA0062CE6C");
			AddGuid("ASF_Simple_Index_Object", "33000890-E5B1-11CF-89F4-00A0C90349CB");
			AddGuid("ASF_Index_Object", "D6E229D3-35DA-11D1-9034-00A0C90349BE");
			AddGuid("ASF_Media_Object_Index_Object", "FEB103F8-12AD-4C64-840F-2A1D2F7AD48C");
			AddGuid("ASF_Timecode_Index_Object", "3CB73FD0-0C4A-4803-953D-EDF7B6228F0C");

			// Header Object
			AddGuid("ASF_File_Properties_Object", "8CABDCA1-A947-11CF-8EE4-00C00C205365");
			AddGuid("ASF_Stream_Properties_Object", "B7DC0791-A9B7-11CF-8EE6-00C00C205365");
			AddGuid("ASF_Header_Extension_Object", "5FBF03B5-A92E-11CF-8EE3-00C00C205365");
			AddGuid("ASF_Codec_List_Object", "86D15240-311D-11D0-A3A4-00A0C90348F6");
			AddGuid("ASF_Script_Command_Object", "1EFB1A30-0B62-11D0-A39B-00A0C90348F6");
			AddGuid("ASF_Marker_Object", "F487CD01-A951-11CF-8EE6-00C00C205365");
			AddGuid("ASF_Bitrate_Mutual_Exclusion_Object", "D6E229DC-35DA-11D1-9034-00A0C90349BE");
			AddGuid("ASF_Error_Correction_Object", "75B22635-668E-11CF-A6D9-00AA0062CE6C");
			AddGuid("ASF_Content_Description_Object", "75B22633-668E-11CF-A6D9-00AA0062CE6C");
			AddGuid("ASF_Extended_Content_Description_Object", "D2D0A440-E307-11D2-97F0-00A0C95EA850");
			AddGuid("ASF_Content_Branding_Object", "2211B3FA-BD23-11D2-B4B7-00A0C955FC6E");
			AddGuid("ASF_Stream_Bitrate_Properties_Object", "7BF875CE-468D-11D1-8D82-006097C9A2B2");
			AddGuid("ASF_Content_Encryption_Object", "2211B3FB-BD23-11D2-B4B7-00A0C955FC6E");
			AddGuid("ASF_Extended_Content_Encryption_Object", "298AE614-2622-4C17-B935-DAE07EE9289C");
			AddGuid("ASF_Digital_Signature_Object", "2211B3FC-BD23-11D2-B4B7-00A0C955FC6E");
			AddGuid("ASF_Padding_Object", "1806D474-CADF-4509-A4BA-9AABCB96AAE8");

			// Header Extension Object
			AddGuid("ASF_Extended_Stream_Properties_Object", "14E6A5CB-C672-4332-8399-A96952065B5A");
			AddGuid("ASF_Advanced_Mutual_Exclusion_Object", "A08649CF-4775-4670-8A16-6E35357566CD");
			AddGuid("ASF_Group_Mutual_Exclusion_Object", "D1465A40-5A79-4338-B71B-E36B8FD6C249");
			AddGuid("ASF_Stream_Prioritization_Object", "D4FED15B-88D3-454F-81F0-ED5C45999E24");
			AddGuid("ASF_Bandwidth_Sharing_Object", "A69609E6-517B-11D2-B6AF-00C04FD908E9");
			AddGuid("ASF_Language_List_Object", "7C4346A9-EFE0-4BFC-B229-393EDE415C85");
			AddGuid("ASF_Metadata_Object", "C5F8CBEA-5BAF-4877-8467-AA8C44FA4CCA");
			AddGuid("ASF_Metadata_Library_Object", "44231C94-9498-49D1-A141-1D134E457054");
			AddGuid("ASF_Index_Parameters_Object", "D6E229DF-35DA-11D1-9034-00A0C90349BE");
			AddGuid("ASF_Media_Object_Index_Parameters_Object", "6B203BAD-3F11-48E4-ACA8-D7613DE2CFA7");
			AddGuid("ASF_Timecode_Index_Parameters_Object", "F55E496D-9797-4B5D-8C8B-604DFE9BFB24");
			AddGuid("ASF_Compatibility_Object", "26F18B5D-4584-47EC-9F5F-0E651F0452C9");
			AddGuid("ASF_Advanced_Content_Encryption_Object", "43058533-6981-49E6-9B74-AD12CB86D58C");

			// Stream Properties Object Stream Type
			AddGuid("ASF_Audio_Media", "F8699E40-5B4D-11CF-A8FD-00805F5C442B");
			AddGuid("ASF_Video_Media", "BC19EFC0-5B4D-11CF-A8FD-00805F5C442B");
			AddGuid("ASF_Command_Media", "59DACFC0-59E6-11D0-A3AC-00A0C90348F6");
			AddGuid("ASF_JFIF_Media", "B61BE100-5B4E-11CF-A8FD-00805F5C442B");
			AddGuid("ASF_Degradable_JPEG_Media", "35907DE0-E415-11CF-A917-00805F5C442B");
			AddGuid("ASF_File_Transfer_Media", "91BD222C-F21C-497A-8B6D-5AA86BFC0185");
			AddGuid("ASF_Binary_Media", "3AFB65E2-47EF-40F2-AC2C-70A90D71D343");
		}
	}
}